define([
    'underscore',
    'backbone',
    'marionette',
    'App',
    'text!modules/appointments/requested/requested-appointment.html',
    'text!modules/page/content/_primary-header.html',
    'text!modules/appointments/requested/_requested-appointment-details.html',

    'modules/appointments/requested/requested-appointment-status-view',
    'modules/appointments/requested/requested-booked-pcp-appointment',
    'modules/appointments/requested/requested-booked-additional-instructions',
    'modules/community-care-request/eligibility-details/requested-appointment-eligibility-view',
    'modules/appointments/requested/requested-appointment-provider-preferences-view',
    'modules/appointments/requested/request-cancellation-control-view',
    'modules/appointments/requested/request-messages-layout-view',
    'modules/appointments/messages/request-messages',
    'modules/appointments/appointments-module',
], function(
    _,
    Backbone,
    Marionette,
    app,
    template,
    _primaryHeaderTemplate,
    _detailsTemplate,
    StatusView,
    BookedPCPAppointment,
    BookedPCPInstructions,
    EligibilityView,
    PreferredProvidersView,
    CancellationView,
    MessagesLayoutView,
    Messages,
    module
) {
    'use strict';

    return Marionette.View.extend({
        className: 'appointments',
        template: _.template(template),
        regions: {
            statusRegion: '#request-appointment-status',
            cancellationRegion: '.request-details-cancellation-section',
            bookedPCPAppointmentRegion: '#booked-pcp-appointment',
            bookedPCPInstructionsRegion: '#booked-pcp-instructions',
            eligibilityRegion: '#request-eligibility-section',
            messagesRegion: '#request-appointment-messages',
            preferredProviderRegion: '#provider-preference-region',
        },
        events: {'click #back-btn': 'routeToAppointments'},
        initialize: function(options) {
            this.model = this.mergeCCRequestIntoAppointmentRequestModel();
            if (this.model.has('bookedAppointment')) {
                this.model.set('bookedAppointment', this.model.get('bookedAppointment').attributes);
                this.model.set('instructionsToVeteran', this.model.get('bookedAppointment').instructionsToVeteran);
            }
            if (!_.isUndefined(options.showFeedbackLink)) {
                this._showFeedbackLink = options.showFeedbackLink;
            }
        },
        onRender: function() {
            var cancellationRegion = this.getRegion('cancellationRegion');
            this.showStatusView();
            if (this.model.get('status') === 'Submitted') {
                this.showCancellationView();
            }
            this.showBookedPCPAppoinment();
            this.showBookedPCPInstructions();
            if (app.isCcEligFeatureEnabled() === false) {
                this.ccShowEligibility();
            }
            this.showMessagesLayoutView();
            this.showPreferredProviders();
            if (this._showFeedbackLink) {
                this.showFeedbackLink();
            }
            this.model.on('sync', function(model) {
                if (model.get('status') !== 'Submitted') {
                    cancellationRegion.empty();
                    this.model.off(null, null, this);
                }
            }, this);

            if (typeof gas !== 'undefined') {
                if (this.model.get('status') === 'Submitted') {
                    gas('send', 'event', 'veteran-appointment', 'appointment-request-successful');
                }
            }
        },
        onDestroy: function() {
            this.model.off(null, null, this);
        },
        routeToAppointments: function() {
            Backbone.history.navigate('appointments', {trigger: true});
        },
        showStatusView: function() {
            var statusView = new StatusView({
                className: 'modal-content-section',
                model: this.model,
            });
            var statusRegion = this.getRegion('statusRegion');

            statusRegion.show(statusView);
        },
        showFeedbackLink: function() {
            this.$el.find('#appointment-feedback-link').removeClass('hidden');
        },
        showCancellationView: function() {
            var cancellationView = new CancellationView({
                model: this.model,
                modelType: 'request',
                elAfterCancellation: this.$el.find('.primary-header h2'),
            });
            var cancellationRegion = this.getRegion('cancellationRegion');

            cancellationRegion.show(cancellationView);
        },
        showMessagesLayoutView: function() {
            var messagesCollection;
            var messagesLayoutView;
            
            if (!this.isDestroyed()) {
                messagesCollection = new Messages();
                messagesCollection.fetch({
                    url: this.model.get('links').get('appointment-request-messages')
                        .get('href'),
                }).done(function() {
                    var messagesRegion = this.getRegion('messagesRegion');
                    if (!this.isDestroyed()) {
                        messagesLayoutView = new MessagesLayoutView({
                            model: this.model,
                            collection: messagesCollection,
                        });

                        messagesRegion.show(messagesLayoutView);
                    }
                }.bind(this));
            }
        },
        
        // when: 
        //  the cc appointment request property exists on the model 
        //  (e.g. the request is being loaded from theh request list)
        // then:
        //  set the properties of ccAppointmentRequest on the request model (vs on this property)
        // else:
        //  use the existing model as the ccAppointmentRequest 
        //  (if they exist) are already on the model (e.g. the request is loaded from submission)
        mergeCCRequestIntoAppointmentRequestModel: function() {
            if (this.model.get('ccAppointmentRequest')) {
                this.model.set(this.model.get('ccAppointmentRequest'));
            }
            return this.model;
        },
        ccShowEligibility: function() {
            var eligibilityView;

            if (!this.model.get('address')) {
                this.model.set('address', '');
            }
            // both conditions are needed due to how requested appointments can be viewed. 
            // either through submission or through clicking on one from the list.
            if (this.model.has('ccAppointmentRequest') || this.model.get('objectType') === 'CCAppointmentRequest') {
                eligibilityView = new EligibilityView({model: this.model});
                this.showChildView('eligibilityRegion', eligibilityView);
            }
        },
        showBookedPCPAppoinment: function() {
            var bookedPCPAppointmentRegion = this.getRegion('bookedPCPAppointmentRegion');

            if (this.model.has('bookedAppointment')) {
                bookedPCPAppointmentRegion.show(new BookedPCPAppointment({model: this.model}));
            }
        },
        showBookedPCPInstructions: function() {
            var bookedPCPInstructionsRegion = this.getRegion('bookedPCPInstructionsRegion');

            if (this.model.has('instructionsToVeteran') &&
                this.model.get('instructionsToVeteran') !== '' &&
                this.model.has('ccAppointmentRequest')) {
                bookedPCPInstructionsRegion.show(new BookedPCPInstructions({model: this.model}));
            }
        },
        showPreferredProviders: function() {
            var preferredProviderRegion = this.getRegion('preferredProviderRegion');

            if (this.model.get('ccAppointmentRequest')) {
                preferredProviderRegion.show(new PreferredProvidersView({model: this.model}));
            }
        },
        templateContext: {
            primaryHeader: _.template(_primaryHeaderTemplate),
            details: _.template(_detailsTemplate),
            EXPRESS_CARE_ID: module.clinicalServicesConstants.EXPRESS_CARE_ID,
        },
    });
});
